;;;
;;; disasm6502.lisp
;;;
;;; Code to disassemble 6502 machine code.
;;;
;;; $Version$
;;;
;;; $Log: disasm6502.lisp,v $
;;; Revision 1.1  2005/08/13 02:30:57  jao
;;; Initial revision
;;;
;;;
;;; 30 October 2000 changed string opcodes to keywords, e.g. "NOP" to :NOP
;;; 
;;; Joseph A. Oswald, III
;;;

;;; The previous version was all too clever, exploiting patterns in the 
;;; operation encoding to fill out the opcode table.
;;; let's just declare all the opcodes with address modes
;;; explicitly.

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defconstant +6502-opcode-list+
    '((0 :BRK :IMPLIED) 
      (1 :ORA :ZP-INDIRECT-X) 
      (5 :ORA :ZERO-PAGE) 
      (6 :ASL :ZERO-PAGE) 
      (8 :PHP :IMPLIED) 
      (9 :ORA :IMMEDIATE) 
      (#x0A :ASL :accumulator) 
      (#x0D :ORA :ABSOLUTE) 
      (#x0E :ASL :ABSOLUTE) 
      
      (#x10 :BPL :BRANCH-RELATIVE) 
      (#x11 :ORA :ZP-INDIRECT-Y) 
      (#x15 :ORA :ZERO-PAGE-X) 
      (#x16 :ASL :ZERO-PAGE-X) 
      (#x18 :CLC :IMPLIED) 
      (#x19 :ORA :ABSOLUTE-Y) 
      (#x1D :ORA :ABSOLUTE-X) 
      (#x1E :ASL :ABSOLUTE-X) 

      (#x20 :JSR :ABSOLUTE) 
      (#x21 :AND :ZP-INDIRECT-X) 
      (#x24 :BIT :ZERO-PAGE) 
      (#x25 :AND :ZERO-PAGE) 
      (#x26 :ROL :ZERO-PAGE) 
      (#x28 :PLP :IMPLIED) 
      (#x29 :AND :IMMEDIATE) 
      (#x2A :ROL :accumulator) 
      (#x2C :BIT :ABSOLUTE) 
      (#x2D :AND :ABSOLUTE) 
      (#x2E :ROL :ABSOLUTE) 
      
      (#x30 :BMI :BRANCH-RELATIVE) 
      (#x31 :AND :ZP-INDIRECT-Y) 
      (#x35 :AND :ZERO-PAGE-X) 
      (#x36 :ROL :ZERO-PAGE-X) 
      (#x38 :SEC :IMPLIED) 
      (#x39 :AND :ABSOLUTE-Y) 
      (#x3D :AND :ABSOLUTE-X) 
      (#x3E :ROL :ABSOLUTE-X) 
      
      (#x40 :RTI :IMPLIED) 
      (#x41 :EOR :ZP-INDIRECT-X) 
      (#x45 :EOR :ZERO-PAGE) 
      (#x46 :LSR :ZERO-PAGE) 
      (#x48 :PHA :IMPLIED) 
      (#x49 :EOR :IMMEDIATE) 
      (#x4A :LSR :accumulator) 
      (#x4C :JMP :ABSOLUTE) 
      (#x4D :EOR :ABSOLUTE) 
      (#x4E :LSR :ABSOLUTE) 
      
      (#x50 :BVC :BRANCH-RELATIVE) 
      (#x51 :EOR :ZP-INDIRECT-Y) 
      (#x55 :EOR :ZERO-PAGE-X) 
      (#x56 :LSR :ZERO-PAGE-X) 
      (#x58 :CLI :IMPLIED) 
      (#x59 :EOR :ABSOLUTE-Y) 
      (#x5D :EOR :ABSOLUTE-X) 
      (#x5E :LSR :ABSOLUTE-X) 
      
      (#x60 :RTS :IMPLIED) 
      (#x61 :ADC :ZP-INDIRECT-X) 
      (#x65 :ADC :ZERO-PAGE) 
      (#x66 :ROR :ZERO-PAGE) 
      (#x68 :PLA :IMPLIED) 
      (#x69 :ADC :IMMEDIATE) 
      (#x6A :ROR :accumulator) 
      (#x6C :JMP :INDIRECT) 
      (#x6D :ADC :ABSOLUTE) 
      (#x6E :ROR :ABSOLUTE) 
      
      (#x70 :BVS :BRANCH-RELATIVE) 
      (#x71 :ADC :ZP-INDIRECT-Y) 
      (#x75 :ADC :ZERO-PAGE-X) 
      (#x76 :ROR :ZERO-PAGE-X) 
      (#x78 :SEI :IMPLIED) 
      (#x79 :ADC :ABSOLUTE-Y) 
      (#x7D :ADC :ABSOLUTE-X) 
      (#x7E :ROR :ABSOLUTE-X) 
      
      (#x81 :STA :ZP-INDIRECT-X) 
      (#x84 :STY :ZERO-PAGE) 
      (#x85 :STA :ZERO-PAGE) 
      (#x86 :STX :ZERO-PAGE) 
      (#x88 :DEY :IMPLIED) 
      (#x8A :TXA :IMPLIED) 
      (#x8C :STY :ABSOLUTE) 
      (#x8D :STA :ABSOLUTE) 
      (#x8E :STX :ABSOLUTE) 
      
      (#x90 :BCC :BRANCH-RELATIVE) 
      (#x91 :STA :ZP-INDIRECT-Y) 
      (#x94 :STY :ZERO-PAGE-X) 
      (#x95 :STA :ZERO-PAGE-X) 
      (#x96 :STX :ZERO-PAGE-Y) 
      (#x98 :TYA :IMPLIED) 
      (#x99 :STA :ABSOLUTE-Y) 
      (#x9A :TXS :IMPLIED) 
      (#x9D :STA :ABSOLUTE-X) 
      
      (#xA0 :LDY :IMMEDIATE) 
      (#xA1 :LDA :ZP-INDIRECT-X) 
      (#xA2 :LDX :IMMEDIATE) 
      (#xA4 :LDY :ZERO-PAGE) 
      (#xA5 :LDA :ZERO-PAGE) 
      (#xA6 :LDX :ZERO-PAGE) 
      (#xA8 :TAY :IMPLIED) 
      (#xA9 :LDA :IMMEDIATE) 
      (#xAA :TAX :IMPLIED) 
      (#xAC :LDY :ABSOLUTE) 
      (#xAD :LDA :ABSOLUTE) 
      (#xAE :LDX :ABSOLUTE) 
      
      (#xB0 :BCS :BRANCH-RELATIVE) 
      (#xB1 :LDA :ZP-INDIRECT-Y) 
      (#xB4 :LDY :ZERO-PAGE-X) 
      (#xB5 :LDA :ZERO-PAGE-X) 
      (#xB6 :LDX :ZERO-PAGE-Y) 
      (#xB8 :CLV :IMPLIED) 
      (#xB9 :LDA :ABSOLUTE-Y) 
      (#xBA :TSX :IMPLIED) 
      (#xBC :LDY :ABSOLUTE-X) 
      (#xBD :LDA :ABSOLUTE-X) 
      (#xBE :LDX :ABSOLUTE-Y) 
      
      (#xC0 :CPY :IMMEDIATE) 
      (#xC1 :CMP :ZP-INDIRECT-X) 
      (#xC4 :CPY :ZERO-PAGE) 
      (#xC5 :CMP :ZERO-PAGE) 
      (#xC6 :DEC :ZERO-PAGE) 
      (#xC8 :INY :IMPLIED) 
      (#xC9 :CMP :IMMEDIATE) 
      (#xCA :DEX :IMPLIED) 
      (#xCC :CPY :ABSOLUTE) 
      (#xCD :CMP :ABSOLUTE) 
      (#xCE :DEC :ABSOLUTE)
      
      (#xD0 :BNE :BRANCH-RELATIVE) 
      (#xD1 :CMP :ZP-INDIRECT-Y) 
      (#xD5 :CMP :ZERO-PAGE-X) 
      (#xD6 :DEC :ZERO-PAGE-X) 
      (#xD8 :CLD :IMPLIED) 
      (#xD9 :CMP :ABSOLUTE-Y) 
      (#xDD :CMP :ABSOLUTE-X) 
      (#xDE :DEC :ABSOLUTE-X) 
      
      (#xE0 :CPX :IMMEDIATE) 
      (#xE1 :SBC :ZP-INDIRECT-X) 
      (#xE4 :CPX :ZERO-PAGE) 
      (#xE5 :SBC :ZERO-PAGE) 
      (#xE6 :INC :ZERO-PAGE) 
      (#xE8 :INX :IMPLIED) 
      (#xE9 :SBC :IMMEDIATE) 
      (#xEA :NOP :IMPLIED) 
      (#xEC :CPX :ABSOLUTE) 
      (#xED :SBC :ABSOLUTE) 
      (#xEE :INC :ABSOLUTE) 

      (#xF0 :BEQ :BRANCH-RELATIVE) 
      (#xF1 :SBC :ZP-INDIRECT-Y) 
      (#xF5 :SBC :ZERO-PAGE-X) 
      (#xF6 :INC :ZERO-PAGE-X) 
      (#xF8 :SED :IMPLIED) 
      (#xF9 :SBC :ABSOLUTE-Y) 
      (#xFD :SBC :ABSOLUTE-X) 
      (#xFE :INC :ABSOLUTE-X))))

#|
;; for later enhancement...
;;
(defconstant +ncr65c02-opcode-list+
  (append +6502-opcode-list+
	  '((#x04 :tsb :zero-page)
	    (#x0c :tsb :absolute)
	    (#x12 :ora :zp-indirect)
	    (#x14 :trb :zero-page)
	    (#x80 :bra :branch-relative)
	    
;; 65c816: note, SEP, REP are immediate, but
;; *always* 2 bytes in length; other immediate shift between
;; one & two byte arguments, depending on the m,x flags
;; BRK takes two bytes
;; no TRB, TSB
|#

#|
;; old..
(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun add-opcodes (decode-vector opcode-string opcode-vector)  
    (map nil #'(lambda (opcode)
                 (setf (aref decode-vector opcode) opcode-string))
         opcode-vector)
    decode-vector)
  
  (defun combine-nibbles (low-nibble high-nibble)
    (+ low-nibble (* 16 high-nibble)))

  (defun combine-nibble-list (low-list high-list)
    (mapcan #'(lambda (low) (mapcar #'(lambda (high) 
                                        (combine-nibbles low high))
                                    high-list))
            low-list))

  (defun accumulator-opcodes (high-nibble)
    (mapcan #'(lambda (low-nibble)
                (list (combine-nibbles low-nibble high-nibble)
                      (combine-nibbles low-nibble (1+ high-nibble))))
            '(#x1 #x5 #x9 #xd)))
          
  (defun load-index-opcodes (immediate-opcode)
    (mapcar #'(lambda (offset)
                (+ immediate-opcode offset))
            '(#x00 #x04 #x0c #x14 #x1c)))
  
  (defun store-index-opcodes (zero-page-opcode)
    (mapcar #'(lambda (offset)
                (+ zero-page-opcode offset))
            '(#x00 #x08 #x10)))
  
  (defun compare-index-opcodes (immediate-opcode)
    (mapcar #'(lambda (offset)
                (+ immediate-opcode offset))
            '(#x00 #x04 #x0c)))
  
  (defun shift-rotate-opcodes (accumulator-opcode)
    (mapcar #'(lambda (offset)
                (+ accumulator-opcode offset))
            '(#x-04 #x00 #x04 #x0c #x14)))

  (defun inc/dec-opcodes (zero-page-opcode)
    (mapcar #'(lambda (offset)
                (+ zero-page-opcode offset))
            '(#x00 #x08 #x10 #x18))))
  
;; old form of 6502 instruction data

(defconstant +6502-mnemonic-assoc+
  (list (cons :ORA (accumulator-opcodes #x0))
        (cons :AND (accumulator-opcodes #x2))
        (cons :EOR (accumulator-opcodes #x4))
        (cons :ADC (accumulator-opcodes #x6))

        (cons :STA (remove #x89 (accumulator-opcodes #x8))) 
        ;; #x89 would be STA immediate, which makes no sense--it is a NOP

        (cons :LDA (accumulator-opcodes #xa))
        (cons :CMP (accumulator-opcodes #xc))
        (cons :SBC (accumulator-opcodes #xe))

        '(:BRK . #x00)
        '(:BPL . #x10)
        '(:BMI . #x30)
        '(:BVC . #x50)
        '(:BVS . #x70)
        '(:BCC . #x90)
        '(:BCS . #xb0)
        '(:BNE . #xd0)
        '(:BEQ . #xf0)

        '(:JSR . #x20)
        '(:RTI . #x40)
        '(:RTS . #x60)

        (cons :LDY (load-index-opcodes #xa0))
        (cons :LDX (load-index-opcodes #xa2))

        (cons :STY (store-index-opcodes #x84))
        (cons :STX (store-index-opcodes #x86))

        (cons :CPY (compare-index-opcodes #xc0))
        (cons :CPX (compare-index-opcodes #xe0))

        (cons :ASL (shift-rotate-opcodes #x0a))
        (cons :ROL (shift-rotate-opcodes #x2a))
        (cons :LSR (shift-rotate-opcodes #x4a))
        (cons :ROR (shift-rotate-opcodes #x6a))

        (cons :INC (inc/dec-opcodes #xe6))
        (cons :DEC (inc/dec-opcodes #xc6))

        ;; (cons "BIT" (inc/dec-opcodes #x24))
        '(:BIT #x24 #x2c)
        
        '(:JMP #x4c #x6c)

        '(:INX . #xe8)
        '(:INY . #xc8)

        '(:DEX . #xca)
        '(:DEY . #x88)

        '(:PHP . #x08)
        '(:PLP . #x28)
        '(:PHA . #x48)
        '(:PLA . #x68)

        '(:TAX . #xaa)
        '(:TXA . #x8a)

        '(:TAY . #xa8)
        '(:TYA . #x98)
        
        '(:TSX . #xba)
        '(:TXS . #x9a)

        '(:CLC . #x18)
        '(:SEC . #x38)
        '(:CLI . #x58)
        '(:SEI . #x78)
        '(:CLV . #xb8)
        '(:CLD . #xd8)
        '(:SED . #xf8)

        '(:NOP . #xea)))

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun lookup-mnemonic (keyword)
    (cdr (assoc keyword +6502-mnemonic-assoc+ :test #'eq))))

(defconstant +6502-address-mode-assoc+
  (list 
   (cons :immediate
         (append (combine-nibble-list 
                  '(#x9)
                  '(#x0 #x2 #x4 #x6 #x8 #xA #xC #xE))
                 (combine-nibble-list
                  '(#x0)
                  '(#xa #xc #xe))
                 (list #xa2)))
                 
   
   '(:indirect . #x6c) ;; JMP indirect

   (cons :implied
         (append (mapcar #'lookup-mnemonic
                         '(:NOP 
                           :BRK
                           :RTS
                           :RTI
                           :INX :INY
                           :DEX :DEY 
                           :PLA :PHA
                           :PLP :PHP
                           :TAX :TXA
                           :TSX :TXS
                           :TAY :TYA
                           :CLC :SEC
                           :CLD :SED
                           :CLI :SEI
                           :CLV))))
   (cons :accumulator
	 (combine-nibble-list '(#xa)
			      '(#x0 #x2 #x4 #x6))) 
   ;; accumulator ASL, ROL, LSR, ROR
   )
                 
   (cons :absolute
         (append 
          (combine-nibble-list '(#xd #xe)
                               '(#x0 #x2 #x4 #x6 #x8 #xa #xc #xe))
          (combine-nibble-list '(#xc)
                               '(#x2 #x4 #x8 #xa #xc #xe))
          (list (lookup-mnemonic :JSR)))
         )
   (cons :absolute-x
         (append
          (combine-nibble-list '(#xd)
                               '(#x1 #x3 #x5 #x7 #x9 #xb #xd #xf))
          (combine-nibble-list '(#xe) 
                               '(#x1 #x3 #x5 #x7 #xd #xf))
          '(#xbc)) ;; LDY absolute,X
         )
   (cons :absolute-y
         (append
          (combine-nibble-list '(#x9)
                               '(#x1 #x3 #x5 #x7 #x9 #xb #xd #xf))
          '(#xbe)) ;; LDX absolute,Y
         )
   (cons :zero-page
         (append
          (combine-nibble-list '(#x5 #x6)
                               '(#x0 #x2 #x4 #x6 #x8 #xa #xc #xe))
          (combine-nibble-list '(#x4)
                               '(#x2 #x8 #xa #xc #xe)))
         )
   (cons :zero-page-x
         (append
          (combine-nibble-list '(#x5)
                               '(#x1 #x3 #x5 #x7 #x9 #xb #xd #xf))
          (combine-nibble-list '(#x6)
                               '(#x1 #x3 #x5 #x7 #xd #xf))
          '(#x94 #xb4)) ;; STY,LDY zpx
         )                          
   (cons :zero-page-y
         '(#x96 #xb6) ;; STX,LDX zpy
         )
   (cons :zp-indirect-x
          (combine-nibble-list '(#x1)
                               '(#x0 #x2 #x4 #x6 #x8 #xa #xc #xe))
          )
   (cons :zp-indirect-y
         (combine-nibble-list '(#x1)
                              '(#x1 #x3 #x5 #x7 #x9 #xb #xd #xf))
         )
   (cons :branch-relative
         (mapcar #'lookup-mnemonic
                 '(:BPL :BMI
                   :BVC :BVS
                   :BCC :BCS
                   :BEQ :BNE)))))
   
|#

#|
;; old

;;; these three routines are pretty similar, I ought to have
;;; written a macro.

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun make-address-mode-lookup (address-assoc)
    (let ((a (make-array 256 :initial-element nil)))
      (labels ((set-list (assoc-element)
                 (let ((mode (car assoc-element))
                       (codes (cdr assoc-element)))
                   (let ((code-list (if (listp codes)
                                      codes
                                      (list codes))))
                     (dolist (c code-list)
                       (setf (aref a c) mode))))))
        (mapcar #'set-list address-assoc)
        a)))
  
  (defun make-mnemonic-lookup (mnemonic-assoc)
    (let ((a (make-array 256 :initial-element :???)))
      (labels ((set-list (assoc-element)
                 (let ((mnemonic (car assoc-element))
                       (codes (cdr assoc-element)))
                   (let ((code-list (if (listp codes)
                                      codes
                                      (list codes))))
                     (dolist (c code-list)
                       (setf (aref a c) mnemonic))))))
        (mapcar #'set-list mnemonic-assoc)
        a)))
  
  (defun make-instruction-length-lookup (address-assoc)
    (let ((a (make-array 256 :initial-element 1)))
      (labels ((set-list (assoc-element)
                 (let ((address-mode (car assoc-element))
                       (codes (cdr assoc-element)))
                   (let ((code-list (if (listp codes)
                                      codes
                                      (list codes))))
                     (dolist (c code-list)
                       (setf (aref a c)
                             (ecase address-mode
                               ((:implied
				 :accumulator) 1)
                               ((:immediate
                                 :branch-relative
                                 :zero-page
                                 :zero-page-x
                                 :zero-page-y
                                 :zp-indirect-x
                                 :zp-indirect-y)
                                2)
                               ((:absolute
                                 :absolute-x
                                 :absolute-y
                                 :indirect)
                                3))))))))
        (mapcar #'set-list address-assoc)
        a))))
                            
(defconstant +6502-mnemonic-lookup+ (make-mnemonic-lookup 
                                     +6502-mnemonic-assoc+))

(defconstant +6502-address-mode-lookup+ (make-address-mode-lookup 
                                         +6502-address-mode-assoc+))

(defconstant +6502-instruction-length-lookup+ (make-instruction-length-lookup
                                               +6502-address-mode-assoc+))

|#

;;; NEW form
(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun make-address-mode-lookup (opcode-list)
    (let ((a (make-array 256 :initial-element nil)))
      (mapcar #'(lambda (opcode-list)
		  (let ((opcode (first opcode-list))
			(mode (third opcode-list)))
		    (setf (aref a opcode)
			  mode)))
	      opcode-list)
      a))

  (defun make-mnemonic-lookup (opcode-list)
    (let ((a (make-array 256 :initial-element :???)))
      (mapcar #'(lambda (opcode-list)
		  (let ((opcode (first opcode-list))
			(mnemonic (second opcode-list)))
		    (setf (aref a opcode)
			  mnemonic)))
	      opcode-list)
      a)))

(defconstant +6502-mnemonic-lookup+
  (make-mnemonic-lookup +6502-opcode-list+))

(defconstant +6502-address-mode-lookup+
  (make-address-mode-lookup +6502-opcode-list+))

(defun 6502-mnemonic (opcode)
  (aref +6502-mnemonic-lookup+ opcode))

(defun 6502-address-mode (opcode)
  (aref +6502-address-mode-lookup+ opcode))

(defun 6502-instruction-length (opcode)
  ;; later, want to dynamically calculate depending
  ;; on 65816 mode
  (let ((mode (6502-address-mode opcode)))
    (ecase mode
      ((:implied :accumulator) 1)
      ((:immediate :branch-relative
		   :zero-page :zero-page-x
		   :zero-page-y	:zp-indirect-x
		   :zp-indirect-y) 2)
      ((:absolute :absolute-x
		  :absolute-y :indirect) 3))))
#|
(defun make-6502-table ()
  (dotimes (i 256)
    (format t "~2X: ~A ~A length: ~D~%" i (aref +6502-mnemonic-lookup+ i)
            (aref +6502-address-mode-lookup+ i)
            (aref +6502-instruction-length-lookup+ i))))
|#

#|
;;; see disasm6502-oo.lisp

;;; somehow, need to handle the following problems?
;;;
;;; 1) if the code-array does not have a complete instruction on the end, 
;;;    perhaps due to the end being a data dump, e.g. say the binary
;;;    bytes FF FF 20, which would be ??? ??? then JSR, but the address
;;;    for the JSR is not in the code-array, we should either 
;;;    flag the error (which will happen, when the aref tries to go off the end)
;;;    or substitute some garbage like "JSR $????" for the disassembly
;;;
;;; 2) similarly, if we were, say disassembling a ROM image from the Apple II,
;;;    and the disassembly ran into the reset-vector area, the bytes at $FFFE..FFFF
;;;    would have some disasssembly that wrapped around to $0. 
;;;
;;; somehow, I think these things probably won't matter too much.
;;;
;;; other goals:
;;;
;;; make an intelligent disassembler, able to create true assembly code?
;;; e.g. accept 1) descriptions of what is code and what are garbage data blocks.
;;;             2) labels of routines, etc., e.g. $FDED = Apple II Character Out "COUT"
;;;             
;;; and generate 1) lists of routine entry points, etc., with automatic label generation
;;;              
;;; to do this, probably ought to keep the addresses & operands in numeric form.
;;;

(defun string-disassemble-6502-instruction (code-array offset address)
  "Decodes a 6502 instruction at OFFSET in CODE-ARRAY, which is presumed
to correspond to a physical ADDRESS.

   Returns a string containing a disassembly of that instruction, 
           the offset of the following instruction, and
           the address of the following instruction (which may wrap around in the 64k
address space of the 6502."

  (let* ((bytes-available (- (length code-array) offset))
         (opcode (aref code-array offset))
         (mnemonic (6502-mnemonic opcode))
         (instruction-length (6502-instruction-length opcode))
         (full-instruction (<= instruction-length bytes-available))
         (disasm-length (min instruction-length bytes-available))
         (address-mode (6502-address-mode opcode)))
    (let ((string-form 
           (with-output-to-string (str)
             (format str "~4,'0X-   " address)
             (dotimes (i disasm-length)
               (format str "~2,'0X " (aref code-array (+ offset i))))

             (let* ((arg-1 (if (>= disasm-length 2)
                             (format nil "~2,'0X" (aref code-array (+ offset 1)))
                             "??"))
                    (arg-2 (if (>= disasm-length 3)
                             (format nil "~4,'0X" 
                                     (+ (aref code-array (+ offset 1))
                                        (* 256 (aref code-array (+ offset 2)))))
                             (format nil "??~A" arg-1))))
               
               (format str "~20T~A" mnemonic)
               
               (ecase address-mode
                 ((:implied nil))
		 (:accumulator (format str "   A" arg-1))
		 (:immediate (format str "   #$~A" arg-1))
		 (:zero-page (format str "   $~A" arg-1))
		 (:zero-page-x (format str "   $~A,X" arg-1))
		 (:zero-page-y (format str "   $~A,Y" arg-1))
		 (:zp-indirect-x (format str "   ($~A,X)" arg-1))
		 (:zp-indirect-y (format str "   ($~A),Y" arg-1))
		 (:absolute (format str "   $~A" arg-2))
		 (:absolute-x (format str "   $~A,X" arg-2))
		 (:absolute-y (format str "   $~A,Y" arg-2))
		 (:indirect (format str "   ($~A)" arg-2))
		 (:branch-relative
                  (if full-instruction
		      (let* ((arg-1 (aref code-array (+ offset 1)))
			     (branch-offset (if (<= arg-1 127)
						arg-1
						(- arg-1 256)))
			     (destination (mod (+ address 2 branch-offset) 
					       #x10000)))
			(format str "   $~4,'0X" destination))
		      (format str "   $????"))))))))
      
      (values string-form
              (+ offset instruction-length)
              (mod (+ address instruction-length) #x10000)))))

(defun string-disasm-6502 (code-array start-offset end-offset start-address
                               &optional (output-stream t))
  (do ((pc start-address)
       (offset start-offset))
      ((>= offset end-offset) nil)
    (multiple-value-bind (asm-line next-offset next-address)
          (string-disassemble-6502-instruction code-array offset pc)
      
      (format output-stream "~A~%" asm-line)
      (setf pc next-address)
      (setf offset next-offset))))

|#

    

